<?php

namespace Televox\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class GraficasLlamadas extends Controller
{
    public function getRegistros(Request $request)
    {
        $results = [];
        $cuantos = 0;
        $titulo = [];

        $datos = $request->validate([
            'tipoLlam' => 'numeric|between:-1,1|nullable',
            'identificador' => 'numeric|nullable',
            'usuario' => 'alpha_num|nullable',
            'nombre' => 'string|nullable',
            'telefono' => 'alpha_num|nullable',
            'cuenta' => 'alpha_num|nullable',
            'segundosinicial' => 'between:0,59|numeric|required_with_all:minutosinicial,horasinicial',
            'minutosinicial' => 'between:0,59|numeric|required_with_all:segundosinicial,horasinicial',
            'horasinicial' => 'between:0,23|numeric|required_with_all:minutosinicial,segundosinicial',
            'segundosfinal' => 'between:0,59|numeric|required_with_all:minutosfinal,horasfinal',
            'minutosfinal' => 'between:0,59|numeric|required_with_all:segundosfinal,horasfinal',
            'horasfinal' => 'between:0,23|numeric|required_with_all:minutosfinal,segundosfinal',
            'inicio' => 'before_or_equal:today|date|required_with_all:fminutosinicial,fhorasinicial,fsegundosinicial',
            'fin' => 'after_or_equal:inicio|before_or_equal:today|date|required_with_all:fminutosfinal,fhorasfinal,fsegundosfinal',
            'fminutosinicial' => 'between:0,59|numeric|required_with_all:fsegundosinicial,fhorasinicial,inicio',
            'fhorasinicial' => 'between:0,23|numeric|required_with_all:fminutosinicial,fsegundosinicial,inicio',
            'fsegundosfinal' => 'between:0,59|numeric|required_with_all:fminutosfinal,fhorasfinal,fin',
            'fminutosfinal' => 'between:0,59|numeric|required_with_all:fsegundosfinal,fhorasfinal,fin',
            'fhorasfinal' => 'between:0,23|numeric|required_with_all:fminutosfinal,fsegundosfinal,fin',
            'status' => 'numeric|nullable',
            'escuchando' => 'boolean|nullable',
            'archive' => 'boolean|required',
            'campaign_id' => 'nullable|string|required_with:list_id,status_id',
            'list_id' => 'nullable|string',
            'status_id' => 'nullable|string',
        ]);

        $recording_log = $datos['archive'] ? 'recording_log_archive' : 'recording_log';

        $filtros = $this->buildFilters($datos, $request);

        $results = $this->fetchResults($datos, $filtros, $recording_log, $request);

        $query = $this->buildQueryLog($datos);
        DB::connection('Televox')->table('tbl_log')->insertGetId([
            'idClientes' => $request->session()->get('idClientes'),
            'ip' => $request->ip(),
            'fecha' => now(),
            'id_usuario' => $request->session()->get('id'),
            'tipo' => 18,
            'detalle' => 'El usuario ' . $request->session()->get('usuario') . " genero graficas con los parametros con: $query"
        ]);

        return ['Estado' => 'ok', 'Registros' => $results];
    }

    private function buildFilters($datos, $request)
    {
        $filtros = [];

        if (!empty($datos['status'])) {
            $filtros[] = ['ts.idStatus', '=', $datos['status']];
        }
        if (!empty($datos['campaign_id'])) {
            $filtros[] = ['vc.campaign_id', '=', $datos['campaign_id']];
            if (!empty($datos['list_id'])) {
                $filtros[] = ['vls.list_id', '=', $datos['list_id']];
            }
            if (!empty($datos['status_id'])) {
                $filtros[] = ['vl.status', '=', $datos['status_id']];
            }
        }
        $filtros[] = ['rl.filename', 'LIKE', '%_' . $request->session()->get('idClientes')];
        if (!empty($datos['cuenta'])) {
            $filtros[] = ['vl.vendor_lead_code', 'LIKE', "%{$datos['cuenta']}%"];
        }
        if (!empty($datos['identificador'])) {
            $filtros[] = ['rl.recording_id', '=', "{$datos['identificador']}"];
        }
        if (!empty($datos['telefono'])) {
            $filtros[] = ['rl.extension', 'LIKE', "%{$datos['telefono']}%"];
        }
        if (!empty($datos['nombre'])) {
            $filtros[] = ['rl.filename', 'LIKE', "%{$datos['nombre']}%"];
        }
        if (!empty($datos['usuario'])) {
            $filtros[] = ['rl.user', 'LIKE', "%{$datos['usuario']}%"];
        }
        if (!empty($datos['segundosinicial'])) {
            $filtros[] = ['rl.length_in_sec', '>=', $datos['segundosinicial'] + $datos['minutosinicial'] * 60 + $datos['horasinicial'] * 3600];
        }
        if (!empty($datos['segundosfinal'])) {
            $filtros[] = ['rl.length_in_sec', '<=', $datos['segundosfinal'] + $datos['minutosfinal'] * 60 + $datos['horasfinal'] * 3600];
        }
        if (!empty($datos['tipoLlam'])) {
            $filtros[] = ['rl.filename', 'LIKE', $datos['tipoLlam'] > 0 ? 'IN_%' : 'OUT_%'];
        }

        return $filtros;
    }

    private function fetchResults($datos, $filtros, $recording_log, $request)
    {
        $query = DB::connection('Televox')->table("$recording_log as rl")
            ->leftJoin('tbl_status_grab AS tsg', 'tsg.id_grab', '=', 'rl.filename')
            ->leftJoin('tbl_status AS ts', 'tsg.id_status', '=', 'ts.idStatus')
            ->leftJoin('audios_listen AS al', 'al.id_grab', '=', 'rl.filename')
            ->where($filtros)
            ->where(function ($q) use ($datos) {
                if (!empty($datos['inicio'])) {
                    $q->whereDate('rl.start_time', '>=', $datos['inicio'])
                        ->whereTime('rl.start_time', '>=', "{$datos['fhorasinicial']}:{$datos['fminutosinicial']}:{$datos['fsegundosinicial']}");
                }
                if (!empty($datos['fin'])) {
                    $q->whereDate('rl.start_time', '<=', $datos['fin'])
                        ->whereTime('rl.start_time', '<=', "{$datos['fhorasfinal']}:{$datos['fminutosfinal']}:{$datos['fsegundosfinal']}");
                }
            })
            ->where(function ($q) use ($request) {
                $q->where('al.id_usuario', '=', $request->session()->get('id'))
                    ->orWhereNull('al.id_usuario');
            });

        if (!empty($datos['campaign_id']) || !empty($datos['cuenta'])) {
            $query->crossJoin('vicidial_list AS vl')
                ->join('vicidial_lists AS vls', 'vls.list_id', '=', 'vl.list_id')
                ->join('vicidial_campaigns AS vc', 'vc.campaign_id', '=', 'vls.campaign_id')
                ->whereRaw('rl.lead_id=vl.lead_id');
        }

        return $query->select(DB::raw('COUNT(rl.recording_id) AS CANT'), DB::raw("COALESCE(ts.nombre,'Sin status') as NombreStatus"))
            ->groupBy('NombreStatus')
            ->get();
    }

    private function buildQueryLog($datos)
    {
        $query = '[';
        foreach ($datos as $key => $val) {
            $query .= "$key => $val,";
        }
        return rtrim($query, ',') . ']';
    }
}
