<?php

namespace Televox\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class EliminarArchivos extends Controller
{
    public function eliminar(Request $request, $id)
    {
        if (!$request->session()->get('eliminar', false)) {
            return [
                'Estado' => 'error',
                'Descripcion' => 'No tiene permisos para eliminar grabaciones'
            ];
        }

        $datos = $request->validate([
            'archive' => 'boolean|required',
        ]);

        $recording_log = $datos['archive'] ? 'recording_log_archive' : 'recording_log';

        $result = DB::connection('Dialvox')
            ->table("$recording_log AS rl")
            ->where('rl.recording_id', $id)
            ->where('rl.filename', 'LIKE', '%_' . $request->session()->get('idClientes'))
            ->select([
                'rl.recording_id AS Nombre',
                DB::raw("date(rl.start_time) as 'FechaInicio'"),
                DB::raw("time(rl.start_time) AS 'HoraInicio'"),
                DB::raw("time(rl.end_time) AS 'HoraFin'"),
                DB::raw("sec_to_time(rl.length_in_sec) AS 'Duracion'"),
                'rl.filename AS GRAB_ID',
                'rl.user',
                DB::raw("substr(rl.location,Locate('MP3/',rl.location)+4) AS 'GRAB_FILE'"),
                'rl.extension AS phone_number',
                'rl.lead_id AS lead_id'
            ])
            ->first();

        if (!$result) {
            return ['Estado' => 'error', 'Descripcion' => 'Grabacion no encontrada'];
        }

        $RUTA_COMPLETA = $this->getRutaCompleta($result);

        if (file_exists($RUTA_COMPLETA) && unlink($RUTA_COMPLETA)) {
            DB::connection('Dialvox')->table($recording_log)->where('recording_id', $result->Nombre)->delete();
            $this->logEliminacion($request, $result);

            return ['Estado' => 'ok'];
        }

        return ['Estado' => 'error', 'Descripcion' => 'No se pudo borrar la grabacion'];
    }

    public function eliminarVarios(Request $request)
    {
        if (!$request->session()->get('eliminar', false)) {
            return ['Estado' => 'error', 'Descripcion' => 'No tiene permisos para eliminar grabaciones'];
        }

        $datos = $request->validate([
            'tipoLlam' => 'numeric|between:-1,1|nullable',
            'identificador' => 'numeric|nullable',
            'usuario' => 'alpha_num|nullable',
            'nombre' => 'string|nullable',
            'telefono' => 'alpha_num|nullable',
            'cuenta' => 'alpha_num|nullable',
            'segundosinicial' => 'between:0,59|numeric|required_with_all:minutosinicial,horasinicial',
            'minutosinicial' => 'between:0,59|numeric|required_with_all:segundosinicial,horasinicial',
            'horasinicial' => 'between:0,23|numeric|required_with_all:minutosinicial,segundosinicial',
            'segundosfinal' => 'between:0,59|numeric|required_with_all:minutosfinal,horasfinal',
            'minutosfinal' => 'between:0,59|numeric|required_with_all:segundosfinal,horasfinal',
            'horasfinal' => 'between:0,23|numeric|required_with_all:minutosfinal,segundosfinal',
            'inicio' => 'before_or_equal:today|date|required_with_all:fminutosinicial,fhorasinicial,fsegundosinicial',
            'fin' => 'after_or_equal:inicio|before_or_equal:today|date|required_with_all:fminutosfinal,fhorasfinal,fsegundosfinal',
            'fminutosinicial' => 'between:0,59|numeric|required_with_all:fminutosinicial,fhorasinicial,inicio',
            'fminutosinicial' => 'between:0,59|numeric|required_with_all:fsegundosinicial,fhorasinicial,inicio',
            'fhorasinicial' => 'between:0,23|numeric|required_with_all:fminutosinicial,fsegundosinicial,inicio',
            'fsegundosfinal' => 'between:0,59|numeric|required_with_all:fminutosfinal,fhorasfinal,fin',
            'fminutosfinal' => 'between:0,59|numeric|required_with_all:fsegundosfinal,fhorasfinal,fin',
            'fhorasfinal' => 'between:0,23|numeric|required_with_all:fminutosfinal,fsegundosfinal,fin',
            'pagina' => 'min:0|numeric|required',
            'status' => 'numeric|nullable',
            'escuchando' => 'boolean|nullable',
            'archive' => 'boolean|required',
            'campaign_id' => 'nullable|string|required_with:list_id,status_id',
            'list_id' => 'nullable|string|required_with:status_id',
            'status_id' => 'nullable|string',
            'ingroup_id' => 'nullable|string',
        ]);

        $recording_log = $datos['archive'] ? 'recording_log_archive' : 'recording_log';

        $filtros = $this->buildFilters($datos, $request);

        $results = $this->getResults($datos, $filtros, $recording_log, $request);

        $errores = $this->deleteFiles($results, $recording_log, $request);

        if (empty($errores)) {
            return ['Estado' => 'ok'];
        } else {
            return ['Estado' => 'error', 'Descripcion' => 'Hubo algunos errores borrando las grabaciones', 'Errores' => $errores];
        }
    }

    private function buildFilters($datos, $request)
    {
        $filtros = [['rl.filename', 'LIKE', '%_' . $request->session()->get('idClientes')]];

        if (!empty($datos['status'])) {
            $filtros[] = ['ts.idStatus', '=', $datos['status']];
        }
        if (!empty($datos['campaign_id'])) {
            $filtros[] = ['vc.campaign_id', '=', $datos['campaign_id']];
            if (!empty($datos['list_id'])) {
                $filtros[] = ['vls.list_id', '=', $datos['list_id']];
            }
            if (!empty($datos['status_id'])) {
                $filtros[] = ['vl.status', '=', $datos['status_id']];
            }
        } elseif (!empty($datos['ingroup_id'])) {
            $filtros[] = ['vc.campaign_id', '=', $datos['ingroup_id']];
        }

        if (!empty($datos['cuenta'])) {
            $filtros[] = ['vl.vendor_lead_code', 'LIKE', "%{$datos['cuenta']}%"];
        }
        if (!empty($datos['identificador'])) {
            $filtros[] = ['rl.recording_id', '=', "{$datos['identificador']}"];
        }
        if (!empty($datos['telefono'])) {
            $filtros[] = ['rl.extension', 'LIKE', "%{$datos['telefono']}%"];
        }
        if (!empty($datos['nombre'])) {
            $filtros[] = ['rl.filename', 'LIKE', "%{$datos['nombre']}%"];
        }
        if (!empty($datos['usuario'])) {
            $filtros[] = ['rl.user', 'LIKE', "%{$datos['usuario']}%"];
        }
        if (!empty($datos['segundosinicial'])) {
            $filtros[] = ['rl.length_in_sec', '>=', $datos['segundosinicial'] + $datos['minutosinicial'] * 60 + $datos['horasinicial'] * 3600];
        }
        if (!empty($datos['segundosfinal'])) {
            $filtros[] = ['rl.length_in_sec', '<=', $datos['segundosfinal'] + $datos['minutosfinal'] * 60 + $datos['horasfinal'] * 3600];
        }
        if (!empty($datos['tipoLlam'])) {
            $filtros[] = ['rl.filename', 'LIKE', $datos['tipoLlam'] > 0 ? 'IN_%' : 'OUT_%'];
        }

        return $filtros;
    }

    private function getResults($datos, $filtros, $recording_log, $request)
    {
        $query = DB::connection('Televox')->table("$recording_log as rl")
            ->leftJoin('tbl_status_grab AS tsg', 'tsg.id_grab', '=', 'rl.filename')
            ->leftJoin('tbl_status AS ts', 'tsg.id_status', '=', 'ts.idStatus')
            ->leftJoin('audios_listen AS al', 'al.id_grab', '=', 'rl.filename')
            ->where($filtros)
            ->where(function ($q) use ($datos) {
                if (!empty($datos['inicio'])) {
                    $q->whereDate('rl.start_time', '>=', date('Y-m-d', strtotime($datos['inicio'])))
                      ->whereTime('rl.start_time', '>=', date('H:i:s', strtotime("{$datos['fhorasinicial']}:{$datos['fminutosinicial']}:{$datos['fminutosinicial']}")));
                }
                if (!empty($datos['fin'])) {
                    $q->whereDate('rl.start_time', '<=', date('Y-m-d', strtotime($datos['fin'])))
                      ->whereTime('rl.start_time', '<=', date('H:i:s', strtotime("{$datos['fhorasfinal']}:{$datos['fminutosfinal']}:{$datos['fminutosfinal']}")));
                }
            })
            ->where(function ($q) use ($request) {
                $q->where('al.id_usuario', '=', $request->session()->get('id'))
                  ->orWhereNull('al.id_usuario');
            });

        if (!empty($datos['campaign_id']) || !empty($datos['cuenta'])) {
            $query->crossJoin('vicidial_list AS vl')
                  ->join('vicidial_lists AS vls', 'vls.list_id', '=', 'vl.list_id')
                  ->join('vicidial_campaigns AS vc', 'vc.campaign_id', '=', 'vls.campaign_id')
                  ->whereRaw('rl.lead_id=vl.lead_id');
        } elseif (!empty($datos['ingroup_id'])) {
            $query->join('vicidial_closer_log AS vc', 'vc.vicidial_id', '=', 'vc.closecallid');
        }

        return $query->limit(1000)
            ->addSelect([
                'rl.recording_id as Nombre',
                DB::raw("date(rl.start_time) as 'FechaInicio'"),
                DB::raw("time(rl.start_time) as 'HoraInicio'"),
                DB::raw("time(rl.end_time) as 'HoraFin'"),
                DB::raw("sec_to_time(rl.length_in_sec) as 'Duracion'"),
                'rl.filename AS GRAB_ID',
                'rl.user',
                DB::raw("substr(rl.location,Locate('MP3/',rl.location)+4) AS 'GRAB_FILE'"),
                'rl.extension AS phone_number',
                'rl.lead_id AS lead_id'
            ])->get();
    }

    private function deleteFiles($results, $recording_log, $request)
    {
        $errores = [];

        foreach ($results as $result) {
            $RUTA_COMPLETA = $this->getRutaCompleta($result);

            if (file_exists($RUTA_COMPLETA) && unlink($RUTA_COMPLETA)) {
                DB::connection('Dialvox')->table($recording_log)->where('recording_id', $result->Nombre)->delete();
                $this->logEliminacion($request, $result);
            } else {
                $errores[] = "No se pudo borrar la grabacion: '$result->GRAB_ID'";
            }
        }

        return $errores;
    }

    private function getRutaCompleta($result)
    {
        $RUTA_COMPLETA = env('RUTAGRAB', '/mnt/Grabaciones') . DIRECTORY_SEPARATOR . $result->GRAB_FILE;
        if (!file_exists($RUTA_COMPLETA)) {
            $RUTA_COMPLETA = env('RUTAGRAB', '/mnt/Grabaciones') . DIRECTORY_SEPARATOR . str_replace('-', '/', $result->FechaInicio) . '/' . str_replace(str_replace('-', '/', $result->FechaInicio) . '/', '', $result->GRAB_FILE);
        }
        return $RUTA_COMPLETA;
    }

    private function logEliminacion($request, $result)
    {
        DB::connection('Televox')->table('tbl_log')->insertGetId([
            'idClientes' => $request->session()->get('idClientes'),
            'ip' => $request->ip(),
            'fecha' => now(),
            'id_usuario' => $request->session()->get('id'),
            'tipo' => 6,
            'detalle' => "El usuario " . $request->session()->get('usuario') . " elimino la grabacion: $result->GRAB_ID Ext/Usuario: $result->user: Fecha: $result->FechaInicio Hora: $result->HoraInicio"
        ]);
    }
}
