<?php

namespace Televox\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class Buscador extends Controller
{
    public function search(Request $req)
    {
        $datos = $req->validate([
            'identificador' => ['numeric', 'nullable'],
            'nombre' => ['string', 'nullable'],
            // Duracion Inicial
            'segundos_duracion_inicial' => ['numeric', 'between:0,59', 'required_with_all:minutos_duracion_inicial,horas_duracion_inicial', 'required'],
            'minutos_duracion_inicial' => ['numeric', 'between:0,59', 'required_with_all:segundos_duracion_inicial,horas_duracion_inicial', 'required'],
            'horas_duracion_inicial' => ['numeric', 'between:0,23', 'required_with_all:segundos_duracion_inicial,minutos_duracion_inicial', 'required'],
            // Duracion final
            'segundos_duracion_final' => ['numeric', 'between:0,59', 'required_with_all:minutos_duracion_final,horas_duracion_final', 'required'],
            'minutos_duracion_final' => ['numeric', 'between:0,59', 'required_with_all:segundos_duracion_final,horas_duracion_final', 'required'],
            'horas_duracion_final' => ['numeric', 'between:0,23', 'required_with_all:segundos_duracion_final,minutos_duracion_final', 'required'],
            // Fechas inicial
            'fecha_inicio' => ['date', 'before_or_equal:today', 'required_with_all:segundo_fecha_inicial,minuto_fecha_inicial,hora_fecha_inicial', 'required'],
            'segundo_fecha_inicial' => ['numeric', 'between:0,59', 'required_with_all:fecha_inicio,minuto_fecha_inicial,hora_fecha_inicial', 'required'],
            'minuto_fecha_inicial' => ['numeric', 'between:0,59', 'required_with_all:fecha_inicio,segundo_fecha_inicial,hora_fecha_inicial', 'required'],
            'hora_fecha_inicial' => ['numeric', 'between:0,23', 'required_with_all:fecha_inicio,segundo_fecha_inicial,minuto_fecha_inicial', 'required'],
            // Fecha final
            'fecha_fin' => ['date', 'before_or_equal:today', 'after_or_equal:fecha_inicio', 'required_with_all:segundo_fecha_final,minuto_fecha_final,hora_fecha_final', 'required'],
            'segundo_fecha_final' => ['numeric', 'between:0,59', 'required_with_all:fecha_fin,minuto_fecha_final,hora_fecha_final', 'required'],
            'minuto_fecha_final' => ['numeric', 'between:0,59', 'required_with_all:fecha_fin,segundo_fecha_final,hora_fecha_final', 'required'],
            'hora_fecha_final' => ['numeric', 'between:0,23', 'required_with_all:fecha_fin,segundo_fecha_final,minuto_fecha_final', 'required'],
            // Otros
            'archivado' => ['boolean', 'required'],
            'pagina' => ['numeric', 'min:1', 'required'],
            // Autenticacion
            'usuario' => ['string', 'regex:/^(?:[A-Za-z0-9+\/]{4})*(?:[A-Za-z0-9+\/]{2}==|[A-Za-z0-9+\/]{3}=)?$/', 'required'],
            'password' => ['string', 'regex:/^(?:[A-Za-z0-9+\/]{4})*(?:[A-Za-z0-9+\/]{2}==|[A-Za-z0-9+\/]{3}=)?$/', 'required'],
        ]);

        $recording_log = $datos['archivado'] ? 'recording_log_archive' : 'recording_log';

        $usuario = DB::connection('Televox')
            ->table('tbl_usuario')
            ->where('usuario', $datos['usuario'])
            ->where('password', $datos['password'])
            ->first();

        if (!$usuario) {
            return response()->json([
                'Estado' => 'Error',
                'Mensaje' => 'Usuario o contraseña incorrecta'
            ], 403);
        }

        $idClientes = $usuario->idClientes;
        $idUsuario = $usuario->Id;
        $nombreUsuario = base64_decode(base64_decode($datos['usuario']));

        $filtros = [
            ['rl.filename', 'LIKE', "%_$idClientes"],
            ['rl.length_in_sec', '>=', $this->convertToSeconds($datos, 'duracion_inicial')],
            ['rl.length_in_sec', '<=', $this->convertToSeconds($datos, 'duracion_final')],
            ['rl.start_time', '>=', $this->convertToDateTime($datos, 'fecha_inicio', 'fecha_inicial')],
            ['rl.end_time', '<=', $this->convertToDateTime($datos, 'fecha_fin', 'fecha_final')]
        ];

        if (isset($datos['identificador'])) {
            $filtros[] = ['recording_id', $datos['identificador']];
        }

        if (isset($datos['nombre'])) {
            $filtros[] = ['rl.filename', 'LIKE', "%{$datos['nombre']}%"];
        }

        $offset = ($datos['pagina'] - 1) * 20;
        $query = DB::connection('Televox')
            ->table("$recording_log as rl")
            ->where($filtros)
            ->addSelect('rl.recording_id as Nombre')
            ->addSelect(DB::raw("date(rl.start_time) as 'Fecha Inicio'"))
            ->addSelect(DB::raw("date(rl.end_time) as 'Fecha Fin'"))
            ->addSelect(DB::raw("time(rl.start_time) as 'Hora Inicio'"))
            ->addSelect(DB::raw("time(rl.end_time) as 'Hora Fin'"))
            ->addSelect(DB::raw("sec_to_time(rl.length_in_sec) as 'Duracion'"))
            ->addSelect("rl.filename as 'Grabacion'")
            ->addSelect("rl.user as 'Usuario'")
            ->addSelect("rl.extension as 'Extension'")
            ->offset($offset)
            ->limit(20);

        $registros = $query->get();
        $cantidad = $query->count();

        $valores = json_encode($datos);
        DB::connection('Televox')
            ->table('tbl_log')
            ->insertGetId([
                'idClientes' => $idClientes,
                'ip' => $req->ip(),
                'fecha' => now(),
                'id_usuario' => $idUsuario,
                'tipo' => 2,
                'detalle' => "El usuario '$nombreUsuario' hizo una consulta a las grabaciones con los parametros: $valores"
            ]);

        return [
            'Estado' => 'Ok',
            'Registros' => $registros,
            'Conteo' => $cantidad
        ];
    }

    private function convertToSeconds($datos, $prefix)
    {
        return $datos["segundos_$prefix"] +
            ($datos["minutos_$prefix"] * 60) +
            ($datos["horas_$prefix"] * 3600);
    }

    private function convertToDateTime($datos, $fechaKey, $horaPrefix)
    {
        $dateUnix = strtotime($datos[$fechaKey]);
        return date('Y-m-d H:i:s', mktime(
            $datos["hora_$horaPrefix"],
            $datos["minuto_$horaPrefix"],
            $datos["segundo_$horaPrefix"],
            date('n', $dateUnix),
            date('j', $dateUnix),
            date('Y', $dateUnix)
        ));
    }
}
